﻿using System;
using System.ComponentModel;
using System.Data;
using System.Web.UI.WebControls;
using System.Web.UI.WebControls.WebParts;
using Microsoft.SharePoint;

namespace BB.SP2010.WebParts.QuickLinks.QuickLinks
{
	[ToolboxItemAttribute(false)]
	public class QuickLinks : WebPart
	{
		private bool useCurrentWeb;
		private string listName;
		private string orderByColumn = "";
		private string defaultText = "";
		private DropDownList ddl;
		private ListItemCollection lic;

		[WebBrowsable(true),
		WebDescription("Yes if list is in current site, false if list is in site collection root."),
		WebDisplayName("Use local list"),
		Personalizable(PersonalizationScope.User)]
		public bool UseCurrentWeb
		{
			set { useCurrentWeb = value; }
			get { return useCurrentWeb; }
		}

		[WebBrowsable(true),
		WebDescription("Name of the list which is populating the drop down menu."),
		WebDisplayName("List name"),
		Personalizable(PersonalizationScope.User)]
		public string ListName
		{
			set { listName = value; }
			get { return listName; }
		}

		[WebBrowsable(true),
		WebDescription("Name of the column which the list will sort by"),
		WebDisplayName("Order By column"),
		Personalizable(PersonalizationScope.User)]
		public string OrderByColumn
		{
			set { orderByColumn = value; }
			get { return orderByColumn; }
		}

		[WebBrowsable(true),
		WebDescription("Text that appears in the drop down menu"),
		WebDisplayName("Default text"),
		Personalizable(PersonalizationScope.User)]
		public string DefaultText
		{
			set { defaultText = value; }
			get { return defaultText; }
		}

		protected override void CreateChildControls()
		{
			ddl = new DropDownList();
			ddl.EnableViewState = true;
			ddl.AutoPostBack = true;
			this.Controls.Add(ddl);

			ddl.DataSource = this.ViewState["QuickLinksDataSource"];
			ddl.DataTextField = "Text";
			ddl.DataValueField = "Value";
			ddl.DataBind();
		}

		protected override void OnLoad(EventArgs e)
		{
			EnsureChildControls();

			if (!this.Page.IsPostBack)
			{
				DataTable dt = new DataTable();
				dt.Columns.Add("Text");
				dt.Columns.Add("Value");

				DataRow dr = dt.NewRow();
				dr["Text"] = defaultText;
				dr["Value"] = String.Empty;
				dt.Rows.Add(dr);

				SPWeb web;

				if (useCurrentWeb)
				{
					web = SPContext.Current.Web;
				}
				else
				{
					web = SPContext.Current.Site.RootWeb;
				}

				SPList quickLinksList;

				try
				{
					quickLinksList = web.Lists[listName];
				}
				catch
				{
					SetErrorMessage(String.Format("{0} list cannot be found", listName));
					return;
				}

				bool orderList = true;

				if (orderByColumn.Trim() == String.Empty)
				{
					orderList = false;
				}

				if (orderList)
				{
					if (!quickLinksList.Fields.ContainsField(orderByColumn))
					{
						SetErrorMessage(String.Format("List does not contain field {0}", orderByColumn));
						return;
					}
					else
					{
						SPField orderByField = quickLinksList.Fields[orderByColumn];
						if (!(orderByField is SPFieldNumber))
						{
							SetErrorMessage(String.Format("{0} is not a Number column", orderByColumn));
							return;
						}
					}
				}

				SPQuery query = new SPQuery();
				if (orderList)
				{
					query.Query = String.Format("<OrderBy><FieldRef Name='{0}' Ascending='TRUE'></FieldRef></OrderBy>", orderByColumn);
				}
				else
				{
					query.Query = String.Empty;
				}

				SPListItemCollection quickLinks = quickLinksList.GetItems(query);

				if (quickLinks.Count > 0)
				{
					foreach (SPListItem quickLinkItem in quickLinks)
					{
						SPFieldUrlValue link = new SPFieldUrlValue(quickLinkItem["URL"].ToString());
						DataRow drListItem = dt.NewRow();
						drListItem["Text"] = link.Description;
						drListItem["Value"] = link.Url;
						dt.Rows.Add(drListItem);
					}
				}

				this.ViewState.Add("QuickLinksDataSource", dt);
			}
			else
			{
				if (ddl.SelectedIndex > 0)
				{
					this.Page.Response.Redirect(ddl.SelectedValue);
				}
			}
		}

		private void SetErrorMessage(string errorMsg)
		{
			lic = new ListItemCollection();
			ListItem liError = new ListItem(errorMsg);
			lic.Add(liError);
			ddl.DataSource = lic;
			ddl.DataBind();
		}
	}
}
